% function used to calculate the analytical solution
% of temperature distribution in an around Borehole 
% Heat Exchangers (BHE) proposed by Beier2014 
%
% Richard A. Beier (2014) Transient heat transfer in a 
% CXA borehole heat exchanger, 
% Geothermics 51: 470-482. 
% 
%
% Author: Chaofan Chen
% Email:  cchaofan1311(at)gmail(dot)com

% set length of the borehole
Len = 50; % unit meter
% set input heat rate
% positive Q means heating up the soil;
% negative Q means cooling down.
Q = -5000; % unit watt
% initial soil temperature
T_s = 25; 

% property of the ground
k_s = 2.50;   % heat conductivity of soil. Unit W/m/K 
c_s = 2.24e6;   % heat capacity of soil. Unit J/m3/K
alpha_s = k_s / c_s; 

% property of the BHE
% borehole radius
r_b = 0.108; % unit m; ni 
% set circulating water rate
w = 8.33e-3; % unit m3/sec
% heat capacity of circulating water
c_f = 4190 * 1000; % unit J/K/m3
% set dimensions and properties of coaxial pipe
d_po = 0.110; % outflow pipe diameter, unit m
d_eo = 0.178; % inflow pipe diameter, unit m
k_pp = 1.3; % heat conductivity of internal pipe, unit W/m/K
k_ep = 1.3; % heat conductivity of external pipe, unit W/m/K
t_e = 5.87e-3; % external pipe wall thickness, unit m
t_i = 7.34e-3; % internal pipe wall thickness, unit m
r_ei = d_eo/2-t_e; %inside of external pipe radius, unit m
r_pi = d_po/2-t_i; %inside of internal pipe radius, unit m

% property of grout
c_g = 3.8e6; % heat capacity of grout. unit J/K/m3
k_g = 0.73; % heat conductivity of grout. unit W/m/K

% calculate the convective film resistance inside pipe
v_po = w/(pi*r_ei*r_ei-pi/4*d_po*d_po); 
v_pi = w/(pi*r_pi*r_pi); 
mu_f = 1.14e-3; % unit kg/m/s; 
rho_f = 1000; % unit kg/m3

% Reynolds number
Re_po = rho_f*v_po*(2*(r_ei-d_po/2))/mu_f; 
Re_pi = rho_f*v_pi*2*r_pi/mu_f;

% estimate film coefficients by the Gnielinski correlation
k_f = 0.59; % unit W/m/K
Pr = 8.09; 
% Churchill correlation for friction factor; Eq.(C.4)
f_po = 2/power(1/(sqrt(power(8/Re_po,10)+power(Re_po/36500,20)))+power(2.21*log(Re_po/7),10),1/5);
f_pi = 2/power(1/(sqrt(power(8/Re_pi,10)+power(Re_pi/36500,20)))+power(2.21*log(Re_pi/7),10),1/5);
% Gnielinski correlation is used to estimate convective film coefficient;
% Eq.(C.1)
Nu_Gl = (f_po/2*(Re_po-1000)*Pr)/(1+12.7*sqrt(f_po/2)*(power(Pr,2/3)-1)); 
Nu_G2 = (f_pi/2*(Re_pi-1000)*Pr)/(1+12.7*sqrt(f_pi/2)*(power(Pr,2/3)-1)); 
Nu_lam = 4.364; 

% evaluate Nusselt number; based on value of Re choose appropriate
% correlation
if Re_po < 2000
    Nu_po = Nu_lam; 
else
    Nu_po = Nu_Gl; 
end

if Re_pi < 2000
    Nu_pi = Nu_lam; 
else
    Nu_pi = Nu_G2; 
end

h_po = k_f * Nu_po/ (2*(r_ei-d_po/2)) ; % unit: W/m2/K
h_pi = k_f * Nu_pi/ (2*r_pi) ; % unit: W/m2/K

% evaluate the shunt resistance; 
R_12 = 1/(2*pi*r_pi*h_pi)+1/(2*pi*k_pp)*log(d_po/2/r_pi)+1/(pi*d_po*h_po); 
% calculate dimensionless shunt conductance; Eq.(9)
N_12 = Len/(R_12*w*c_f);

% calculate dimensionless flow areas
A_D1 = (r_ei^2-(d_po/2)^2)/(d_eo/2)^2;
A_D2 = (r_pi/(d_eo/2))^2;

% calculate conductance at grout/pipe interface for outer pipe; Eq.(13)
R_g = 1/(pi*2*r_ei*h_po)+1/(2*pi*k_ep)*log((d_eo/2)/r_ei);
N_g = Len/(w*c_f*R_g); 
% calculate the ratio of grout an ground thermal conductivities; 
kappa = k_g/k_s;
% calculate ratio of volumetric heat capacities;
H_g = c_g/c_s; 
% calculate ratio of water and soil heat capacities
H_f = c_f/c_s; 
% calculate dimensionless soil conuctance
N_s = 2*pi*k_s*Len/(w*c_f); 
% calculate dimensionless borehole radius
r_Db = r_b/(d_eo/2); 


% to evaluate the dimensionless fluid inlet and outlet temperatures, 
% set zD = 0; 
z_D = 0.0; 
% values of r_D and z_Dsand correspond to the location chosen to evaluate
% the temperature in the grout. 
% values of r_D and z_Dsand correspond to the location chosen to evaluate 
% the temperature in the ground (sand). 
% set the location to evaluate temperature in soil
z_Dsand = 0.50; 
r_D = r_Db; 
r_Dsand = (r_b + 0.024)/(d_eo/2); % this is problematic

% calculate the dimensionless time tD1
ndiv = 10; 
ncycles = 5; 
timeDstart = 0.02; 
xincr = 1.0/ndiv; 
id = [0:ndiv]';
expon_id = xincr * id; 
timeD = zeros(ndiv+1,1);
for i=1:ndiv+1
    timeDfirst(i,1) = 10^expon_id(i,1) * timeDstart; 
end

% calculate dimensionless temperature as a function of dimensionless time
% TD1 = dimensionless temperature
% DD1 = dimensionless temperature derivative
% tD1 = dimensionless time (Fourier number)
RD = result(N_s,N_g,N_12,H_g,H_f,kappa,A_D1,A_D2,r_Db,r_Dsand,z_D,z_Dsand,timeDfirst);

% convert from dimensionless Temperature to the real temperature
% TD1 and TD2 are the dimensionless temperature
TD1 = RD(:,3);
TD2 = RD(:,4);
% tD1 is the dimensionless time
tD1 = RD(:,2);
TDS1 = RD(:,5);
TGS1 = RD(:,6);


% evaluate the time and temperatures from correspnding 
% dimensionless variables
t  = tD1 * c_s * (d_eo/2)^2 / k_s; % this is in seconds
T1 = Q * TD1 / (2 * pi * k_s * Len) + T_s; 
T2 = Q * TD2 / (2 * pi * k_s * Len) + T_s; 

T_avg = ( T1 + T2 ) / 2; 
TS1 = Q * TDS1 / (2 * pi * k_s * Len) + T_s; 
TG1 = Q * TGS1 / (2 * pi * k_s * Len) + T_s; 

% organize the plot
% semilogx(t/3600,T_avg); 
% semilogx(t/3600,TS1);
% semilogx(t/3600,TG1);

% organize the result in a matrix R
R = [t, T1, T2, TS1, TG1];

% write the result into the txt file. 
dlmwrite('result_mat_R.txt', R); 

% outflow temperature
figure(1);
plot(t/3600/24,T2);
hold on;

time = 3000;
TimeD = k_s * time/(c_s*(d_eo/2)^2);

% calculate dimensionless temperature as a function of dimensionless time
% TD1 = dimensionless temperature
% DD1 = dimensionless temperature derivative
% tD1 = dimensionless time (Fourier number)
Rz = zresult(N_s,N_g,N_12,H_g,H_f,kappa,A_D1,A_D2,r_Db,r_D,z_D,TimeD);

% convert from dimensionlessnm Temperature to the real temperature
% TD1z and TD2z are the dimensionless temperature
TD1z = Rz(:,3);
TD2z = Rz(:,4);
zD = Rz(:,2);
TDSz = Rz(:,5);
TDGz = Rz(:,6);

nz=40;
for iz = 1:nz;
    T1z(iz,1) = Q * TD1z(iz,1) / (2 * pi * k_s * Len) + T_s;
    T2z(iz,1) = Q * TD2z(iz,1) / (2 * pi * k_s * Len) + T_s;
    TSz(iz,1) = Q * TDSz(iz,1) / (2 * pi * k_s * Len) + T_s; 
    TGz(iz,1) = Q * TDGz(iz,1) / (2 * pi * k_s * Len) + T_s; 
    H(iz,1)= Len*zD(iz,1);
end

% organize the result of vertical temperature in a matrix R
Rz = [H, T1z, T2z, TGz, TSz];

% write the result into the txt file. 
dlmwrite('result_mat_Rz.txt', Rz); 

% temperature
figure(2);
plot(T1z,H,T2z,H);
ylim([0,70]);
set(gca,'Ydir','rev');







